import os
import time
import psutil
import subprocess
from typing import Optional, Tuple, Dict, Any
import requests
from urllib.parse import urlencode, urljoin
import json
import sys
import gzip
from datetime import datetime
import configparser


# Program version
PROG_VERSION = "PWFER.1.0.1"  

# Configuration Parameters
WORKER_SCRIPT = "PlanoWFEWorker.py"  # Path to your worker script
HEARTBEAT_FILE = "PlanoWFEWorker_Heartbeat.txt"  # Heartbeat file generated by the worker
RUNNER_HEARTBEAT_FILE = "PlanoWFERunner_Heartbeat.txt"  # Heartbeat file generated by the runner
HEARTBEAT_TIMEOUT = 10*60  # Timeout in seconds to check for heartbeat
PROCESS_TERMINATION_TIMEOUT = 3  # Timeout in seconds for process termination
PROCESS_TERMINATION_RETRIES = 3  # Number of retries to ensure process termination
TERMINATION_RETRY_DELAY = 0.5  # Delay in seconds between termination retries
TOUCH_RETRY_TIMEOUT = 10  # Total time to retry in seconds
TOUCH_RETRY_DELAY = 0.5  # Delay between retries in seconds


#config = configparser.ConfigParser()
#config.read('wfe_config.ini')
#WFA_API_BASE_URL = config['WFE']['API_BASE_URL']
#WFA_API_TOKEN = config['WFE']['API_TOKEN']
#PROXY_URL = config['system']['PROXY_URL']

# Configuration
if os.getenv("COMPUTERNAME") == "DF":   # Dialog Factory
    WFA_API_BASE_URL = "https://df.workforce-elements.com/api/v1/"
    WFA_API_TOKEN = "API92c53168b304be03b4b8d1178e72155dd77de2ed668be260468888ef38a28f43"
    PROXY_URL = ""
    PROXY = None
    PY_PATH = "D:\\Professional Workforce\\venv\\Scripts\\python.exe"
else:
    WFA_API_BASE_URL = "https://test.workforce-elements.com/api/v1/"
    WFA_API_TOKEN = "API08ecb8b26cf9ee12b0168ca2a57b09baf1a845c44f5c1d9413e01d7ea20a8229"
    PROXY_URL = "http://schnittstelle:!!Planopunkt2023@192.168.200.1:3128"
    PROXY = {
        "http": PROXY_URL,
        "https": PROXY_URL,
    }
    PY_PATH = "D:\\Professional Workforce\\venv\\venv\\Scripts\\python.exe"


def WFE_callAPI(
        dB: str,
        method: str, 
        endpoint: str, 
        sessionID: str = '',
        apiToken: str = '',
        inputParams: Optional[Dict[str, Any]] = None,
        fileName: Optional[str] = None,
        gzip_compress: bool = False,
        jsonBody: Optional[Dict[str, Any]] = None
        ) -> Tuple[int, str, Optional[Any]]:

    httpMethod = method.upper()
    valid_methods = {"GET", "POST", "PATCH", "DELETE"}

    if httpMethod not in valid_methods:
        return 1, "Error: Unsupported HTTP method.", None    

    if not endpoint:
        return 2, "Error: Endpoint must not be empty.", None
    
    if fileName and jsonBody:
        return 3, "Error: fileName and jsonBody cannot be used together.", None
    
    if sessionID and apiToken:
        return 4, "Error: sessionID and apiToken cannot be used together.", None

    base_url = WFA_API_BASE_URL.rstrip("/") + "/"
    url = urljoin(base_url, endpoint.lstrip("/"))

    query_params = {'db': dB} if dB else {}
    query_params.update(inputParams or {})
    if sessionID:
        query_params['sessionId'] = sessionID
    if apiToken:
        query_params['apiToken'] = apiToken

    headers = {}
    files = None

    if fileName:
        try:
            if gzip_compress:
                gzip_file = f"{fileName}.gz"
                with open(fileName, 'rb') as f_in, gzip.open(gzip_file, 'wb') as f_out:
                    f_out.writelines(f_in)
                f_out.close()
                fileName = gzip_file
                headers['Content-Encoding'] = 'gzip'
                headers['Content-Type'] = 'application/octet-stream'
            else:
                headers['Content-Type'] = 'text/csv'

            # Open the file here and keep the file object
            file = open(fileName, 'rb')  # Keep the file open
            files = {'file': (fileName, file, 'application/octet-stream')}

        except Exception as file_err:
            return 5, f"File processing error occurred: {file_err}", None
    else:
        headers['Content-Type'] = 'application/json'
        

    try:
        if files:
            with open(fileName, 'rb') as f:
                if PROXY is None:
                    response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, data=f, timeout=10)
                else:
                    response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, data=f, timeout=10, proxies=PROXY)
        elif jsonBody:
            headers['Content-Type'] = 'application/json'
            if PROXY is None:
                response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, json=jsonBody, timeout=10)
            else:
                response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, json=jsonBody, timeout=10, proxies=PROXY)
        else:
            if PROXY is None:
                response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, timeout=10)
            else:
                response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, timeout=10, proxies=PROXY)

        response.raise_for_status()  # This will raise an exception for HTTP error codes

        if not response.headers.get('Content-Type', '').startswith('application/json'):
            return 6, f"Unexpected Content-Type: {response.headers.get('Content-Type')}", None

    except Exception as err1:
        try:
            data = response.json()
            errorCode = data.get('errorCode', 'unknown')
            resultCode = data.get('resultCode', 'unknown')
            errorMessage = data.get('errorMessage', 'unknown')
            return 7, f"HTTP error {response.status_code} occurred ({err1}). Result Code: {resultCode}, Application Error Message: {errorMessage}", None
        except Exception as err2:
            return 7, f"HTTP error {response.status_code} occurred ({err1}). Additional Error: {err2}", None
    finally:
        # Close the file if it was opened
        if files and 'file' in files:
            files['file'][1].close()

    try:
        data = response.json()

    except json.JSONDecodeError as json_err:
        return 12, f"JSON decode error occurred: {json_err}", None
    except Exception as err:
        return 13, f"Other error occurred: {err}", None

    return 0, "", data



# signal worker state to remote server
#
def signal_worker_state(status, errorNumber, message):

    if os.path.exists(HEARTBEAT_FILE):
        last_modified = datetime.fromtimestamp(os.path.getmtime(HEARTBEAT_FILE)).strftime('%Y-%m-%d %H:%M:%S')
    else:
        last_modified = "N/A"


    attached_data = json.dumps({
        "progVersion": PROG_VERSION,
        "heartbeatFileLastModified": last_modified
    })
    rc, error_details, response = WFE_callAPI("", "POST", "worker/remote/log", apiToken=WFA_API_TOKEN,
        inputParams={
            "category": "PlanoWFERunner", 
            "subcategory": "STATUS", 
            "status": status, 
            "errorNumber": errorNumber, 
            "message": message, 
            "attachedData": attached_data
        })
    print(f"Status: {status}, ErrorNumber: {errorNumber}, Message: '{message}', rc: {rc}, error_details: '{error_details}'")



def serialize_processes(processes):
    serialized = []
    for proc in processes:
        try:
            proc_info = proc.as_dict(attrs=['pid', 'name', 'cmdline'])
            serialized.append(proc_info)
        except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
            continue
    return serialized


# Check if the worker script is running.
#
def workers_running(worker_script):
    worker_processes = []

    try:
        for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
            try:
                cmdline = proc.info['cmdline']
                if cmdline and worker_script in cmdline:
                    # Check if the process has any children running python.exe
                    try:
                        children = proc.children(recursive=True)
                        if any("python.exe" in child.name().lower() for child in children):
                            print(f"Skipping process with PID: {proc.info['pid']} as it has python.exe children")
                            continue
                    except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
                        # If we can't check children, assume it's a valid process
                        pass

                    worker_processes.append(proc)
            except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
                continue
    except Exception as e:
        print(f"Error while checking for running workers: {e}")
        # Return an empty list rather than None in case of error
        return []

    return worker_processes



# Kill the worker process and ensure it is terminated.
#
def kill_worker(proc):
    try:
        proc.terminate()
        proc.wait(timeout=PROCESS_TERMINATION_TIMEOUT)
    except psutil.NoSuchProcess:
        pass
    except psutil.TimeoutExpired:
        proc.kill()

    # Ensure the process is terminated
    for _ in range(PROCESS_TERMINATION_RETRIES):
        if proc.is_running():
            try:
                proc.kill()
                time.sleep(TERMINATION_RETRY_DELAY)
            except psutil.NoSuchProcess:
                break



# Start the worker script as a detached background process.
#
def start_worker(worker_script):
    try:
        # Open null files for redirection
        with open(os.devnull, 'w') as null_out, open(os.devnull, 'r') as null_in:
            process = subprocess.Popen(
                [PY_PATH, worker_script],
                creationflags=subprocess.CREATE_NO_WINDOW | subprocess.DETACHED_PROCESS, #  | subprocess.CREATE_NEW_PROCESS_GROUP
                stdin=null_in,  # Redirect stdin to null
                stdout=null_out, # Redirect stdout to null
                stderr=subprocess.STDOUT # Redirect stderr to stdout (which is null)
            )
        return 0, "", process.pid
    except Exception as e:
        return 1, str(e), -1



# Check if the worker script is responsive.
#
def is_worker_responsive(heartbeat_file):
    try:
        if not os.path.exists(heartbeat_file):
            return False
        last_modified = os.path.getmtime(heartbeat_file)
        return (time.time() - last_modified) < HEARTBEAT_TIMEOUT  # Check if updated in the last timeout seconds
    except Exception:
        return False



# Touch the heartbeat file to update the timestamp.
#
def touch_heartbeat_file(heartbeat_file: str):
    start_time = time.time()

    while time.time() - start_time < TOUCH_RETRY_TIMEOUT:
        try:
            with open(heartbeat_file, 'a'):  # Open in append mode to create if not exists
                os.utime(heartbeat_file, None)  # Update to the current timestamp
            return True
        except (OSError, IOError):
            time.sleep(TOUCH_RETRY_DELAY)

    return False


# Create or update the runner heartbeat file to mark the runner as active.
# This function creates the file "PlanoWFERunner_Heartbeat.txt" if it doesn't exist 
# or updates its timestamp if it does exist.
#
# Returns:
#     bool: True if the heartbeat file was successfully updated, False otherwise.
def touchRunnerHeartbeat() -> bool:
    try:
        return touch_heartbeat_file(RUNNER_HEARTBEAT_FILE)
    except Exception as e:
        print(f"Error touching runner heartbeat file: {str(e)}")
        return False



# Monitor the worker script and take necessary actions.
def monitor_worker():
    try:

        if not psutil.WINDOWS:
            signal_worker_state(0, 1, "This script is only supported on Microsoft Windows.")
            sys.exit(1)

        if len(sys.argv) < 2:
            command = "start"
        else:
            command = sys.argv[1]


        if command == "stop":
            worker_procs = workers_running(WORKER_SCRIPT)
            if worker_procs:
                for proc in worker_procs:
                    kill_worker(proc)
                signal_worker_state(10, 0, "Worker successfully stopped.")
            else:
                signal_worker_state(11, 0, "No worker running to stop.")
                sys.exit(1)
            return


        worker_procs = workers_running(WORKER_SCRIPT)

        if worker_procs == None or len(worker_procs) == 0:

            signal_worker_state(1, 1, "Worker is not running.")
            error_code, error_message, pid = start_worker(WORKER_SCRIPT)
            if error_code == 0:
                signal_worker_state(2, 0, f"The worker was successfully started. PID: {pid}")
            else:
                signal_worker_state(2, 1, "The worker could not be started: " + error_message)
                sys.exit(2)

        elif len(worker_procs) > 1:
            
            signal_worker_state(3, 1, "Multiple worker instances are running.")
            for proc in worker_procs:
                kill_worker(proc)
            signal_worker_state(4, 0, "All worker instances successfully stopped.")
            time.sleep(5)
            error_code, error_message, pid = start_worker(WORKER_SCRIPT)
            if error_code == 0:
                signal_worker_state(5, 0, f"The worker was successfully restarted. PID: {pid}")
            else:
                signal_worker_state(5, 1, "The worker could not be started: " + error_message)
                sys.exit(3)

        else:

            worker_proc = worker_procs[0]
            if not is_worker_responsive(HEARTBEAT_FILE):
                signal_worker_state(5, 1, "Worker is unresponsive.")
                kill_worker(worker_proc)
                if not worker_proc.is_running():
                    signal_worker_state(7, 0, "Unresponsive worker successfully killed.")
                    error_code, error_message, pid = start_worker(WORKER_SCRIPT)
                    if error_code == 0:
                        signal_worker_state(8, 0, f"The worker was successfully restarted. PID: {pid}")
                    else:
                        signal_worker_state(8, 1, "The worker could not be started: " + error_message)
                        sys.exit(4)
                else:
                    signal_worker_state(7, 1, "Unresponsive worker could not be killed.")
                    sys.exit(5)
            else:
                signal_worker_state(6, 0, "Worker is running and responsive.")

    except Exception as e:
        signal_worker_state(9, 1, f"Error during monitoring: {str(e)}")
        sys.exit(6)



if __name__ == "__main__":
    # Touch the runner heartbeat file immediately at script start
    touchRunnerHeartbeat()
    monitor_worker()
