import os
import time
from typing import Optional, Tuple, Dict, Any, Union, List
import requests
from urllib.parse import urlencode, urljoin
import json
import sys
import gzip
from datetime import datetime
import uuid
import subprocess
import tempfile
import traceback
import logging
from contextlib import contextmanager

# Program version
PROG_VERSION = "PWFER.1.0.1"  

# Configuration Parameters
TOUCH_RETRY_TIMEOUT = 10  # Total time to retry in seconds
TOUCH_RETRY_DELAY = 0.5  # Delay between retries in seconds
HEARTBEAT_FILE = "PlanoWFEWorker_Heartbeat.txt"  # Heartbeat file generated by the worker
HEARTBEAT_INTERVAL = 60  # Seconds between updating heartbeat file
PING_INTERVAL = 60  # Seconds between sending ping to backend
API_RETRY_COUNT = 3  # Number of retry attempts for API calls
API_RETRY_DELAY = 10  # Seconds to wait between API retry attempts
JOB_POLL_INTERVAL = 10  # Seconds to wait between job polling

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("PlanoWFEWorker.log"),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger("PlanoWFEWorker")

# Configuration
if os.getenv("COMPUTERNAME") == "DF":   # Dialog Factory
    WFA_API_BASE_URL = "https://df.workforce-elements.com/api/v1/"
    WFA_API_TOKEN = "API92c53168b304be03b4b8d1178e72155dd77de2ed668be260468888ef38a28f43"
    PROXY_URL = ""
    PROXY = None
else:
    WFA_API_BASE_URL = "https://test.workforce-elements.com/api/v1/"
    WFA_API_TOKEN = "API08ecb8b26cf9ee12b0168ca2a57b09baf1a845c44f5c1d9413e01d7ea20a8229"
    PROXY_URL = "http://schnittstelle:!!Planopunkt2023@192.168.200.1:3128"
    PROXY = {
        "http": PROXY_URL,
        "https": PROXY_URL,
    }

# Tool and directory paths
ROSTER_TOOL_EXE = "D:\\plano\\Tools\\RosterTool\\RosterTool.exe"
TEMP_DIR = "D:\\Professional Workforce\\temp"

@contextmanager
def exception_handler(context: str, fatal: bool = False) -> None:
    """
    Context manager to handle exceptions safely without terminating the program.
    
    Args:
        context: Description of the context where the exception occurred
        fatal: If True, re-raise the exception after logging
    """
    try:
        yield
    except Exception as e:
        error_message = f"Exception in {context}: {str(e)}"
        logger.error(error_message)
        logger.error(traceback.format_exc())
        signal_worker_state("ERROR", 1, 999, error_message)
        
        if fatal:
            raise


def verify_environment() -> bool:
    """
    Verify that all required tools and directories exist for the worker to function.
    
    Returns:
        bool: True if environment is validated successfully, False otherwise
    """
    environment_ok = True
    
    # Check if ROSTER_TOOL_EXE exists
    with exception_handler("verify_environment - checking RosterTool"):
        if not os.path.isfile(ROSTER_TOOL_EXE):
            logger.error(f"ERROR: Roster Tool not found at {ROSTER_TOOL_EXE}")
            signal_worker_state("STARTUP", 1, 1, f"Roster Tool not found at {ROSTER_TOOL_EXE}")
            environment_ok = False
    
    # Check if TEMP_DIR exists, create if missing
    with exception_handler("verify_environment - checking temp directory"):
        if not os.path.isdir(TEMP_DIR):
            try:
                os.makedirs(TEMP_DIR, exist_ok=True)
                logger.info(f"Created missing temp directory: {TEMP_DIR}")
                signal_worker_state("STARTUP", 0, 0, f"Created missing temp directory: {TEMP_DIR}")
            except Exception as e:
                logger.error(f"ERROR: Could not create temp directory {TEMP_DIR}: {e}")
                signal_worker_state("STARTUP", 1, 2, f"Could not create temp directory {TEMP_DIR}: {e}")
                environment_ok = False
    
    return environment_ok


def WFE_callAPI(
        dB: str,
        method: str, 
        endpoint: str, 
        sessionID: str = '',
        apiToken: str = '',
        inputParams: Optional[Dict[str, Any]] = None,
        fileName: Optional[str] = None,
        gzip_compress: bool = False,
        jsonBody: Optional[Dict[str, Any]] = None
        ) -> Tuple[int, str, Optional[Any]]:
    """
    Call the WFE API with provided parameters.
    
    Args:
        dB: Database identifier
        method: HTTP method (GET, POST, PATCH, DELETE)
        endpoint: API endpoint path
        sessionID: Session ID for authentication
        apiToken: API token for authentication
        inputParams: Query parameters
        fileName: Path to file to upload
        gzip_compress: Whether to compress the file with gzip
        jsonBody: JSON body for request
    
    Returns:
        Tuple containing:
        - int: Result code (0 for success, non-zero for failure)
        - str: Error message if any
        - Any: Response data if successful, None otherwise
    """
    with exception_handler(f"WFE_callAPI - {method} {endpoint}"):
        httpMethod = method.upper()
        valid_methods = {"GET", "POST", "PATCH", "DELETE"}

        if httpMethod not in valid_methods:
            return 1, "Error: Unsupported HTTP method.", None    

        if not endpoint:
            return 2, "Error: Endpoint must not be empty.", None
        
        if fileName and jsonBody:
            return 3, "Error: fileName and jsonBody cannot be used together.", None
        
        if sessionID and apiToken:
            return 4, "Error: sessionID and apiToken cannot be used together.", None

        base_url = WFA_API_BASE_URL.rstrip("/") + "/"
        url = urljoin(base_url, endpoint.lstrip("/"))

        query_params = {'db': dB} if dB else {}
        query_params.update(inputParams or {})
        if sessionID:
            query_params['sessionId'] = sessionID
        if apiToken:
            query_params['apiToken'] = apiToken

        headers = {}
        files = None
        file = None

        try:
            if fileName:
                try:
                    if gzip_compress:
                        gzip_file = f"{fileName}.gz"
                        with open(fileName, 'rb') as f_in, gzip.open(gzip_file, 'wb') as f_out:
                            f_out.writelines(f_in)
                            f_out.close()
                        fileName = gzip_file
                        headers['Content-Encoding'] = 'gzip'
                        headers['Content-Type'] = 'application/octet-stream'
                    else:
                        headers['Content-Type'] = 'text/csv'

                    # Open the file here and keep the file object
                    file = open(fileName, 'rb')
                    files = {'file': (fileName, file, 'application/octet-stream')}

                except Exception as file_err:
                    return 5, f"File processing error occurred: {file_err}", None
            else:
                headers['Content-Type'] = 'application/json'
                
            try:
                if files:
                    with open(fileName, 'rb') as f:
                        response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, 
                                                   data=f, timeout=10, proxies=PROXY)
                elif jsonBody:
                    headers['Content-Type'] = 'application/json'
                    response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, 
                                               json=jsonBody, timeout=10, proxies=PROXY)
                else:
                    response = requests.request(method=httpMethod, url=url, params=query_params, headers=headers, 
                                               timeout=10, proxies=PROXY)

                response.raise_for_status()

                if not response.headers.get('Content-Type', '').startswith('application/json'):
                    return 6, f"Unexpected Content-Type: {response.headers.get('Content-Type')}", None

            except requests.exceptions.RequestException as err1:
                try:
                    data = response.json()
                    errorCode = data.get('errorCode', 'unknown')
                    resultCode = data.get('resultCode', 'unknown')
                    errorMessage = data.get('errorMessage', 'unknown')
                    return 7, f"HTTP error {response.status_code} occurred ({err1}). Result Code: {resultCode}, Application Error Message: {errorMessage}", None
                except Exception as err2:
                    return 7, f"HTTP error {response.status_code} occurred ({err1}). Additional Error: {err2}", None
            
            try:
                data = response.json()
            except json.JSONDecodeError as json_err:
                return 12, f"JSON decode error occurred: {json_err}", None
            except Exception as err:
                return 13, f"Other error occurred: {err}", None

            return 0, "", data
        
        finally:
            # Close the file if it was opened
            if file:
                try:
                    file.close()
                except Exception:
                    pass  # Ignore errors when closing


def touch_heartbeat_file(heartbeat_file: str) -> bool:
    """
    Update the timestamp on the heartbeat file or create it if it doesn't exist.
    
    Args:
        heartbeat_file: Path to the heartbeat file
        
    Returns:
        bool: True if successful, False otherwise
    """
    with exception_handler("touch_heartbeat_file"):
        start_time = time.time()
        
        while time.time() - start_time < TOUCH_RETRY_TIMEOUT:
            try:
                with open(heartbeat_file, 'a'):  # Open in append mode to create if not exists
                    os.utime(heartbeat_file, None)  # Update to the current timestamp
                return True
            except (OSError, IOError) as e:
                logger.warning(f"Failed to touch heartbeat file, retrying: {e}")
                time.sleep(TOUCH_RETRY_DELAY)
        
        logger.error("Failed to touch heartbeat file after all retries")
        return False


def signal_worker_state(subcategory: str, status: int, errorNumber: int, message: str) -> None:
    """
    Signal worker state to the remote system.
    
    Args:
        subcategory: Category of the log message
        status: Status code (0 for success/info, non-zero for warnings/errors)
        errorNumber: Error number for additional classification
        message: Descriptive message
    """
    with exception_handler("signal_worker_state"):
        try:
            if os.path.exists(HEARTBEAT_FILE):
                last_modified = datetime.fromtimestamp(os.path.getmtime(HEARTBEAT_FILE)).strftime('%Y-%m-%d %H:%M:%S')
            else:
                last_modified = "N/A"
        except Exception as e:
            logger.error(f"Error getting heartbeat file timestamp: {e}")
            last_modified = "Error"

        attached_data = json.dumps({
            "progVersion": PROG_VERSION,
            "heartbeatFileLastModified": last_modified
        })
        
        for attempt in range(API_RETRY_COUNT):
            try:
                rc, error_details, response = WFE_callAPI("", "POST", "worker/remote/log", apiToken=WFA_API_TOKEN,
                    inputParams={
                        "category": "PlanoWFEWorker", 
                        "subcategory": subcategory, 
                        "status": status, 
                        "errorNumber": errorNumber, 
                        "message": message, 
                        "attachedData": attached_data
                    })
                
                if rc != 0:
                    logger.warning(f"API call failed (attempt {attempt+1}/{API_RETRY_COUNT}): Status: {status}, ErrorNumber: {errorNumber}, Message: '{message}', Error: {error_details}")
                    if attempt < API_RETRY_COUNT - 1:
                        time.sleep(API_RETRY_DELAY)
                        continue
                else:
                    logger.info(f"Status: {status}, ErrorNumber: {errorNumber}, Message: '{message}' sent successfully")
                    break
            except Exception as e:
                logger.error(f"Exception in API call (attempt {attempt+1}/{API_RETRY_COUNT}): {e}")
                if attempt < API_RETRY_COUNT - 1:
                    time.sleep(API_RETRY_DELAY)
                    continue


def accept_job(task_id: int) -> int:
    """
    Send acceptance of a job to the backend.
    
    Args:
        task_id: ID of the task to accept
        
    Returns:
        int: 0 if successful, 1 if failed
    """
    with exception_handler("accept_job"):
        for attempt in range(API_RETRY_COUNT):
            try:
                rc, error_details, response = WFE_callAPI("", "POST", "worker/remote/acceptjob", 
                                                        apiToken=WFA_API_TOKEN, 
                                                        inputParams={"taskId": task_id})
                
                if rc != 0:
                    logger.warning(f"Failed to accept job {task_id} (attempt {attempt+1}/{API_RETRY_COUNT}): {error_details}")
                    if attempt < API_RETRY_COUNT - 1:
                        time.sleep(API_RETRY_DELAY)
                        continue
                    signal_worker_state("JOB", 1, rc, f"Error accepting job: {error_details}")
                    return 1
                else:
                    signal_worker_state("JOB", 0, 0, f"Accepted job: {task_id}")
                    return 0
            except Exception as e:
                logger.error(f"Exception while accepting job (attempt {attempt+1}/{API_RETRY_COUNT}): {e}")
                if attempt < API_RETRY_COUNT - 1:
                    time.sleep(API_RETRY_DELAY)
                
        # If we get here, all attempts failed
        signal_worker_state("JOB", 1, 999, f"All attempts to accept job {task_id} failed")
        return 1


def finish_job(task_id: int) -> int:
    """
    Mark a job as finished in the backend.
    
    Args:
        task_id: ID of the task to finish
        
    Returns:
        int: 0 if successful, 1 if failed
    """
    with exception_handler("finish_job"):
        for attempt in range(API_RETRY_COUNT):
            try:
                rc, error_details, response = WFE_callAPI("", "POST", "worker/remote/finishjob", 
                                                        apiToken=WFA_API_TOKEN, 
                                                        inputParams={"taskId": task_id})
                
                if rc != 0:
                    logger.warning(f"Failed to finish job {task_id} (attempt {attempt+1}/{API_RETRY_COUNT}): {error_details}")
                    if attempt < API_RETRY_COUNT - 1:
                        time.sleep(API_RETRY_DELAY)
                        continue
                    signal_worker_state("JOB", 1, rc, f"Error finishing job: {error_details}")
                    return 1
                else:
                    signal_worker_state("JOB", 0, 0, f"Finished job: {task_id}")
                    return 0
            except Exception as e:
                logger.error(f"Exception while finishing job (attempt {attempt+1}/{API_RETRY_COUNT}): {e}")
                if attempt < API_RETRY_COUNT - 1:
                    time.sleep(API_RETRY_DELAY)
                
        # If we get here, all attempts failed
        signal_worker_state("JOB", 1, 999, f"All attempts to finish job {task_id} failed")
        return 1


def get_next_job() -> Tuple[int, str, Optional[Dict[str, Any]]]:
    """
    Fetch the next job from the backend system.
    
    Returns:
        Tuple containing:
        - int: Task ID if a task is available, 0 otherwise
        - str: Task type string if available, empty string otherwise
        - Optional[Dict[str, Any]]: Task details if available, None otherwise
    """
    with exception_handler("get_next_job"):
        # Add retry logic for network errors
        for attempt in range(API_RETRY_COUNT):
            try:
                rc, error_details, response = WFE_callAPI("", "GET", "worker/remote/nextjob", apiToken=WFA_API_TOKEN)
                
                # If there's an error that's not network-related
                if rc != 0:
                    # Only retry for connection or timeout errors
                    if "Connection error" in error_details or "Timeout error" in error_details or "Connection refused" in error_details:
                        if attempt < API_RETRY_COUNT - 1:
                            logger.warning(f"Network error getting next job (attempt {attempt+1}/{API_RETRY_COUNT}): {error_details}, retrying in {API_RETRY_DELAY}s")
                            time.sleep(API_RETRY_DELAY)
                            continue
                    
                    signal_worker_state("JOB", 1, rc, f"Error getting next job: {error_details}")
                    return 0, "", None

                # Validate the response
                if not response:
                    return 0, "", None

                if not isinstance(response, dict):
                    signal_worker_state("JOB", 2, 2, "Response is not a dictionary")
                    return 0, "", None

                taskId = response.get("taskId")
                if not isinstance(taskId, int):
                    signal_worker_state("JOB", 3, 3, "The 'taskId' property is missing or not an integer")
                    return 0, "", None

                task = response.get("task")
                if not isinstance(task, dict):
                    signal_worker_state("JOB", 4, 4, f"Task {taskId}: The 'task' property is not an object")
                    return 0, "", None

                task_type = task.get("taskType")
                if not isinstance(task_type, str):
                    signal_worker_state("JOB", 5, 5, f"Task {taskId}: The 'taskType' property is missing or not a string")
                    return 0, "", None

                if task_type != "NO_TASK":
                    logger.info(f"Task {taskId} received: {task_type}")
                    signal_worker_state("JOB", 0, 0, f"Task {taskId} received: {task_type}")

                return taskId, task_type, task
                
            except Exception as e:
                # Handle unexpected errors in the retry loop
                if attempt < API_RETRY_COUNT - 1:
                    logger.warning(f"Exception getting next job (attempt {attempt+1}/{API_RETRY_COUNT}): {e}, retrying in {API_RETRY_DELAY}s")
                    time.sleep(API_RETRY_DELAY)
                else:
                    logger.error(f"All {API_RETRY_COUNT} attempts to get next job failed: {e}")
                    signal_worker_state("JOB", 1, 999, f"All {API_RETRY_COUNT} attempts to get next job failed: {str(e)}")
                    return 0, "", None
        
        # This should never be reached, but just in case
        return 0, "", None


def call_roster_tool(file_path: str, temp_dir: str) -> Tuple[int, str, str]:
    """
    Call the external Roster Tool with provided parameters.
    
    Args:
        file_path: Path to the file to process
        temp_dir: Directory to use as working directory
        
    Returns:
        Tuple containing:
        - int: Return code from the roster tool
        - str: Standard output from the tool
        - str: Standard error from the tool
    """
    with exception_handler("call_roster_tool"):
        command = [ROSTER_TOOL_EXE, "import", "msm", file_path, ";"]
        logger.info(f"Calling Roster Tool with command: {command}")

        try:
            result = subprocess.run(
                command,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                cwd=temp_dir,
                timeout=300  # Add a timeout of 5 minutes
            )
            logger.info(f"Roster Tool finished with return code {result.returncode}")
            signal_worker_state("JOB", 0, 0, f"Roster Tool: rc={result.returncode}, stdout: {result.stdout[:100]}..., stderr: {result.stderr[:100]}...")
            return result.returncode, result.stdout, result.stderr
        except FileNotFoundError:
            logger.error(f"Roster Tool Executable not found at {ROSTER_TOOL_EXE}")
            signal_worker_state("JOB", 1, 1, "Roster Tool Executable not found")
            return 1, "", "Executable not found"
        except subprocess.TimeoutExpired:
            logger.error("Roster Tool timed out after 5 minutes")
            signal_worker_state("JOB", 1, 2, "Roster Tool timed out after 5 minutes")
            return 2, "", "Process timed out"
        except Exception as e:
            logger.error(f"Roster Tool Exception: {e}")
            signal_worker_state("JOB", 1, 3, f"Roster Tool Exception: {e}")
            return 3, "", str(e)


def generate_unique_temp_filename() -> str:
    """
    Generate a random unique filename with a .csv extension.
    
    Returns:
        str: Unique filename
    """
    return f"{uuid.uuid4()}.csv"


def handleMinTargetMax(task_id: int, task: Dict[str, Any]) -> int:
    """
    Handle a MIN_TARGET_MAX task type.
    
    Args:
        task_id: ID of the task
        task: Task details dictionary
        
    Returns:
        int: 0 if successful, non-zero otherwise
    """
    with exception_handler("handleMinTargetMax"):
        customerId = task.get("customerId")
        tenantId = task.get("tenantId")
        userId = task.get("userId")
        dateFrom = task.get("dateFrom")
        dateTo = task.get("dateTo")
        planoExportId = task.get("planoExportId")
        planoExportsHistId = task.get("planoExportsHistId")
        csvFileContent = task.get("csvFileContent")

        if not csvFileContent:
            logger.error(f"Task {task_id}: Missing csvFileContent")
            signal_worker_state("JOB", 1, 10, f"Task {task_id}: Missing csvFileContent")
            return 10

        # Create a temporary file manually and close it
        try:
            fd, csv_file_path = tempfile.mkstemp(dir=TEMP_DIR, suffix=".csv", text=True)
            try:
                with os.fdopen(fd, "w", encoding="utf-8", newline='') as file:
                    file.write("From date;From time;To date;To time;Group;Time type;Min;Target;Max\n")
                    file.write(csvFileContent)
                    file.flush()  # Ensure data is written to disk

                # Call the external tool after closing the file
                rc, output_std, output_err = call_roster_tool(os.path.basename(csv_file_path), TEMP_DIR)
                logger.info(f"Roster Tool: rc={rc}, stdout: {output_std[:100]}..., stderr: {output_err[:100]}...")
                
                if rc != 0:
                    signal_worker_state("JOB", 1, rc, f"Roster Tool failed with rc={rc}")
                    return rc
                    
            finally:
                # Ensure the file is deleted
                try:
                    os.remove(csv_file_path)
                except Exception as e:
                    logger.warning(f"Failed to remove temporary file {csv_file_path}: {e}")
            
        except Exception as e:
            logger.error(f"Error in handleMinTargetMax: {e}")
            signal_worker_state("JOB", 1, 11, f"Error processing MIN_TARGET_MAX task: {e}")
            return 11

        return 0


def handleDownloadFile(task_id: int, task: Dict[str, Any]) -> int:
    """
    Handle a DOWNLOAD_FILE task type.
    
    Args:
        task_id: ID of the task
        task: Task details dictionary
        
    Returns:
        int: 0 if successful, non-zero otherwise
    """
    with exception_handler("handleDownloadFile"):
        # Example implementation - replace with actual logic
        logger.info(f"Download file task received: {task_id}")
        signal_worker_state("JOB", 0, 0, f"Download file task handled: {task_id}")
        return 0


def handleUploadFile(task_id: int, task: Dict[str, Any]) -> int:
    """
    Handle an UPLOAD_FILE task type.
    
    Args:
        task_id: ID of the task
        task: Task details dictionary
        
    Returns:
        int: 0 if successful, non-zero otherwise
    """
    with exception_handler("handleUploadFile"):
        # Example implementation - replace with actual logic
        logger.info(f"Upload file task received: {task_id}")
        signal_worker_state("JOB", 0, 0, f"Upload file task handled: {task_id}")
        return 0


def handlePingTask(task_id: int, task: Dict[str, Any]) -> int:
    """
    Handle a PING task type.
    
    Args:
        task_id: ID of the task
        task: Task details dictionary
        
    Returns:
        int: 0 if successful, non-zero otherwise
    """
    with exception_handler("handlePingTask"):
        logger.info(f"Ping task received: {task_id}")
        signal_worker_state("PING", 0, 0, f"Worker Ping Response to task {task_id}")
        return 0


def run_worker() -> None:
    """
    Main worker loop that runs continuously.
    """
    last_ping_time = 0
    last_heartbeat_time = 0

    # Signal that we're starting up
    logger.info(f"Worker starting up, version {PROG_VERSION}")
    signal_worker_state("STARTUP", 0, 0, f"Worker starting up, version {PROG_VERSION}")

    # Verify the environment and start main loop if everything is OK
    if verify_environment():
        logger.info("Environment OK, starting worker main loop")
        signal_worker_state("STARTUP", 0, 0, "Environment OK, starting worker main loop")
        
        while True:
            try:
                current_time = time.time()
                
                # Touch the heartbeat file to update the timestamp periodically
                if current_time - last_heartbeat_time >= HEARTBEAT_INTERVAL:
                    with exception_handler("Main loop - heartbeat update"):
                        if touch_heartbeat_file(HEARTBEAT_FILE):
                            last_heartbeat_time = current_time
                            logger.debug("Heartbeat file updated")

                # Send "PING" only once per PING_INTERVAL
                if current_time - last_ping_time >= PING_INTERVAL:
                    with exception_handler("Main loop - periodic ping"):
                        signal_worker_state("PING", 0, 0, "Worker Ping")
                        last_ping_time = current_time
                        logger.debug("Periodic ping sent")

                # Get the next job to process
                with exception_handler("Main loop - get next job"):
                    task_id, task_type, task = get_next_job()

                    # Accept the job if it's a real task
                    if task_type and task_type != "NO_TASK":
                        logger.info(f"Accepting job {task_id} of type {task_type}")
                        accept_job(task_id)
                        
                        # Process the task based on its type
                        with exception_handler(f"Main loop - processing {task_type} task"):
                            result = 0
                            
                            if task_type == "MIN_TARGET_MAX":
                                result = handleMinTargetMax(task_id, task)
                            elif task_type == "DOWNLOAD_FILE":
                                result = handleDownloadFile(task_id, task)
                            elif task_type == "UPLOAD_FILE":
                                result = handleUploadFile(task_id, task)
                            elif task_type == "PING":
                                result = handlePingTask(task_id, task)
                            else:
                                logger.warning(f"Unknown task type: {task_type}")
                                signal_worker_state("JOB", 1, 20, f"Unknown task type: {task_type}")
                            
                            # Mark the job as finished
                            if task_type != "NO_TASK":
                                logger.info(f"Finishing job {task_id}")
                                finish_job(task_id)

            except Exception as e:
                # Log the error but don't terminate
                logger.error(f"Error in main worker loop: {e}")
                logger.error(traceback.format_exc())
                signal_worker_state("ERROR", 1, 999, f"Caught exception in main loop: {str(e)}")
                
                # Wait a bit before continuing to prevent rapid error loops
                time.sleep(30)

            # Sleep for a short period before the next iteration
            time.sleep(JOB_POLL_INTERVAL)


if __name__ == "__main__":
    # Start the worker main loop
    run_worker()
